"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.activate = activate;
exports.deactivate = deactivate;
const vscode = __importStar(require("vscode"));
const path = __importStar(require("path"));
const syncPlan_1 = require("./commands/syncPlan");
const data_1 = require("./data");
const i18n_1 = require("./i18n");
const performance_1 = require("./utils/performance");
const markdownFormatter_1 = require("./markdownFormatter");
const logger_1 = require("./logger");
/**
 * 扩展激活入口
 */
function activate(context) {
    // 初始化日志系统
    logger_1.Logger.initialize('Lexical Markdown Editor');
    logger_1.Logger.info('Extension is now active');
    // 注册 Custom Editor Provider
    const provider = new LexicalMarkdownEditorProvider(context);
    context.subscriptions.push(vscode.window.registerCustomEditorProvider('lexicalMarkdown.editor', provider, {
        webviewOptions: {
            retainContextWhenHidden: true,
        },
        supportsMultipleEditorsPerDocument: false,
    }));
    // 注册 Markdown 格式化命令，用于在保存 .plan-original.md 时预先格式化
    context.subscriptions.push(vscode.commands.registerCommand('lexicalMarkdown.formatMarkdown', async (markdown) => {
        try {
            const formatted = await (0, markdownFormatter_1.formatMarkdown)(markdown);
            console.log('[Lexical Formatter] Successfully formatted markdown');
            return formatted;
        }
        catch (error) {
            console.error('[Lexical Formatter] Failed to format markdown:', error);
            return markdown; // 失败时返回原内容
        }
    }));
    // 注册打开 Lexical Editor 的命令
    const openEditorCommand = vscode.commands.registerCommand('lexicalMarkdown.openEditor', async (param) => {
        let targetUri;
        if (param.uri) {
            // 如果传入了 URI，直接使用
            targetUri = param.uri;
        }
        else {
            // 否则使用当前活动编辑器的文档
            const activeEditor = vscode.window.activeTextEditor;
            if (!activeEditor) {
                vscode.window.showWarningMessage((0, i18n_1.t)('openMarkdownFirst'));
                return;
            }
            const document = activeEditor.document;
            if (document.languageId !== 'markdown') {
                vscode.window.showWarningMessage((0, i18n_1.t)('notMarkdownFile', { languageId: document.languageId }));
                return;
            }
            targetUri = document.uri;
        }
        // 提取 conversationId（用于查找已存在的 webview）
        const conversationId = extractConversationIdFromUri(targetUri);
        // 检查是否已有打开的 webview
        const existingPanel = conversationId
            ? LexicalMarkdownEditorProvider.webviewPanels.get(conversationId)
            : null;
        if (existingPanel && param.part) {
            // 如果 webview 已存在且需要滚动，直接发送滚动消息
            logger_1.Logger.debug('Webview already exists, sending scroll command', { part: param.part });
            existingPanel.webview.postMessage({
                type: 'scrollToPart',
                part: param.part
            });
            // 聚焦到该 webview
            existingPanel.reveal();
        }
        else {
            // 首次打开或无需滚动，保存 part 参数供初始化使用
            context.workspaceState.update('lexical.scrollToPart', param.part);
            // 使用 Custom Editor 打开目标文档
            // 注意：由于 supportsMultipleEditorsPerDocument: false
            // VS Code 会自动处理重复打开的情况，切换到已存在的编辑器而不是创建新实例
            await vscode.commands.executeCommand('vscode.openWith', targetUri, 'lexicalMarkdown.editor');
        }
    });
    // 注册 Plan 刷新命令
    const reloadPlanCommand = new syncPlan_1.ReloadPlanCommand(context);
    const reloadPlanDisposable = vscode.commands.registerCommand(reloadPlanCommand.id, (args) => reloadPlanCommand.execute(args));
    context.subscriptions.push(openEditorCommand, reloadPlanDisposable, { dispose: () => logger_1.Logger.dispose() });
}
/**
 * 扩展停用
 */
function deactivate() { }
/**
 * 从 URI 中提取 conversationId
 * 使用 path 模块解析路径，适配 Windows、Linux、macOS 等各种操作系统
 */
function extractConversationIdFromUri(uri) {
    const filePath = uri.fsPath;
    // 检查文件名是否为 plan.md
    if (path.basename(filePath) !== 'plan.md') {
        return null;
    }
    // 获取 plan.md 的父目录（即 conversationId 目录）
    const parentDir = path.dirname(filePath);
    const conversationId = path.basename(parentDir);
    // 验证父目录的父目录是否为 plans
    const grandParentDir = path.dirname(parentDir);
    if (path.basename(grandParentDir) !== 'plans') {
        return null;
    }
    // 验证 conversationId 格式（可选，增加安全性）
    if (!/^[a-zA-Z0-9_-]+$/.test(conversationId)) {
        return null;
    }
    return conversationId;
}
/**
 * Lexical Markdown Custom Editor Provider
 * 实现 CustomTextEditorProvider 接口
 */
class LexicalMarkdownEditorProvider {
    constructor(context) {
        this.context = context;
        // 存储文件监听器，用于监听 plan.md 的变化
        this.fileWatchers = new Map();
        // 防抖定时器，避免频繁触发文件变化事件
        this.debounceTimers = new Map();
        // 缓存上一次的完整 PlanResult 数据，用于比较是否真正变化
        this.lastPlanDataCache = new Map();
    }
    /**
     * 核心方法：当用户打开 .md 文件时被调用
     */
    async resolveCustomTextEditor(document, webviewPanel, _token) {
        // 开始整体计时
        performance_1.PerformanceTimer.start('Extension/resolveCustomTextEditor');
        performance_1.PerformanceTimer.start('Extension/configureWebview');
        // 配置 webview
        webviewPanel.webview.options = {
            enableScripts: true,
            localResourceRoots: [
                vscode.Uri.joinPath(this.context.extensionUri, 'webview', 'dist')
            ]
        };
        performance_1.PerformanceTimer.end('Extension/configureWebview');
        // 加载 TODO 和 Design Config
        const conversationId = this.extractConversationId(document.fileName);
        const markdownContent = document.getText();
        let planData = {
            id: conversationId || '',
            name: '',
            overview: '',
            status: 'prepare',
            hasConfirmed: false,
            content: markdownContent,
            design: undefined,
            todolist: [],
            parts: []
        };
        // 初始化 diff 数据
        let initialDiffResult = null;
        if (conversationId) {
            try {
                // 计时：加载 Plan 数据
                planData = await (0, performance_1.withTiming)('Extension/loadPlanData', async () => {
                    return await syncPlan_1.PlanDataLoader.loadPlan(this.context, conversationId);
                });
                logger_1.Logger.info('Loaded plan data', {
                    conversationId,
                    todoCount: planData.todolist.length,
                    hasDesign: !!planData.design,
                    contentLength: planData.content.length
                });
                // 初始化 PlanData 缓存（用于文件变化时的数据比较）
                this.lastPlanDataCache.set(conversationId, planData);
                // 计时：计算初始 diff
                initialDiffResult = await (0, performance_1.withTiming)('Extension/calculateDiff', async () => {
                    return await (0, data_1.calculateCurrentDiff)(this.getGenieExtensionPath(), conversationId);
                });
                if (initialDiffResult) {
                    logger_1.Logger.info('Calculated initial diff', {
                        conversationId,
                        stats: initialDiffResult.stats
                    });
                }
                // 注册 webview panel 到 ReloadPlanCommand，以便接收刷新通知
                syncPlan_1.ReloadPlanCommand.registerWebviewPanel(conversationId, webviewPanel);
                logger_1.Logger.debug('WebviewPanel registered', { conversationId });
                // 注册到静态 Map，用于滚动功能
                LexicalMarkdownEditorProvider.webviewPanels.set(conversationId, webviewPanel);
            }
            catch (error) {
                logger_1.Logger.error('Failed to load plan data', error);
            }
        }
        // 计时：生成 HTML
        performance_1.PerformanceTimer.start('Extension/generateHTML');
        webviewPanel.webview.html = this.getHtmlForWebview(webviewPanel.webview, document, planData, initialDiffResult);
        performance_1.PerformanceTimer.end('Extension/generateHTML');
        // 结束整体计时
        performance_1.PerformanceTimer.end('Extension/resolveCustomTextEditor');
        // 监听 webview 消息
        const messageSubscription = webviewPanel.webview.onDidReceiveMessage(async (message) => {
            switch (message.type) {
                case 'switchToMarkdown':
                    // 切换到 Markdown 源码编辑模式
                    await this.switchToMarkdownSource(document, webviewPanel);
                    break;
                case 'switchToPreview':
                    // 切换回 Preview 模式，关闭源码编辑器
                    await this.switchToPreviewMode(document);
                    break;
                case 'buildPlan':
                    await this.handleBuildPlan(message.id);
                    break;
                case 'updatePlanContent':
                    // 处理 Plan 内容统一更新（包含 content、design、todolist）
                    const planResult = message.planResult;
                    const cachedPlanData = this.lastPlanDataCache.get(planResult.id);
                    // 比较 content、todolist、design 是否有变化
                    const hasContentChanged = cachedPlanData?.content !== planResult.content;
                    const hasTodolistChanged = JSON.stringify(cachedPlanData?.todolist) !== JSON.stringify(planResult.todolist);
                    const hasDesignChanged = JSON.stringify(cachedPlanData?.design) !== JSON.stringify(planResult.design);
                    if (hasContentChanged || hasTodolistChanged || hasDesignChanged) {
                        // 更新缓存
                        this.lastPlanDataCache.set(planResult.id, planResult);
                        // 立即计算 diff 并发送给 webview（使用内存数据，无需等待 file watcher）
                        if (hasContentChanged && conversationId) {
                            const mainExtensionPath = this.getGenieExtensionPath();
                            const originalContent = await (0, data_1.getOriginalContent)(mainExtensionPath, conversationId);
                            if (originalContent) {
                                const diffResult = (0, data_1.calculateTextDiff)(originalContent, planResult.content);
                                webviewPanel.webview.postMessage({
                                    type: 'updateDiff',
                                    diffStats: diffResult.stats,
                                    diffChanges: diffResult.changes
                                });
                            }
                        }
                        // 异步保存文件
                        await this.handlePlanContentUpdate(planResult.id, planResult);
                    }
                    break;
                case 'sendDiffToChat':
                    // 发送 diff 信息到 Chat
                    try {
                        if (message.conversationId && message.conversationId !== conversationId) {
                            console.warn('[Lexical Editor] Conversation id mismatch:', message.conversationId, conversationId);
                            return;
                        }
                        if (conversationId && message.diffContent) {
                            this.planActionReport('sendDiff', message.conversationId);
                            await vscode.commands.executeCommand('tencentcloud.codingcopilot.ide.addToChat', {
                                prompt: message.diffContent,
                                options: {
                                    autoSend: false
                                }
                            });
                        }
                    }
                    catch (error) {
                        logger_1.Logger.error('Failed to send diff to chat', error);
                    }
                    break;
                case 'revertToOriginal':
                    // 撤销修改，恢复到原始版本
                    try {
                        if (message.conversationId && message.conversationId !== conversationId) {
                            logger_1.Logger.warn('Conversation id mismatch', {
                                expected: conversationId,
                                received: message.conversationId
                            });
                            return;
                        }
                        if (conversationId) {
                            const mainExtensionPath = this.getGenieExtensionPath();
                            const originalContent = await (0, data_1.getOriginalContent)(mainExtensionPath, conversationId);
                            if (!originalContent) {
                                logger_1.Logger.warn('No original content found to revert', { conversationId });
                                return;
                            }
                            console.log('[Lexical Editor] Reverted to original content');
                            // 重新加载完整的 PlanData 并更新缓存
                            const planData = await (0, data_1.getLocalPlanResult)(mainExtensionPath, conversationId);
                            planData.content = originalContent;
                            // 更新缓存
                            this.lastPlanDataCache.set(conversationId, planData);
                            // 恢复原始版本，diff 为空
                            await webviewPanel.webview.postMessage({
                                type: 'updateMarkdownContent',
                                content: originalContent,
                                diffStats: { additions: 0, deletions: 0, totalChanges: 0 },
                                diffChanges: []
                            });
                            // 异步保存文件
                            this.handlePlanContentUpdate(conversationId, planData);
                        }
                    }
                    catch (error) {
                        logger_1.Logger.error('Failed to revert to original', error);
                    }
                    break;
            }
        });
        // 设置文件监听器，监听 plan.md 的变化
        if (conversationId) {
            await this.setupFileWatcher(conversationId, webviewPanel);
        }
        // 清理
        webviewPanel.onDidDispose(() => {
            // changeDocumentSubscription.dispose();
            messageSubscription.dispose();
            // 从 Map 中移除
            if (conversationId) {
                LexicalMarkdownEditorProvider.webviewPanels.delete(conversationId);
                logger_1.Logger.debug('WebviewPanel disposed', { conversationId });
                // 清理防抖定时器
                const timer = this.debounceTimers.get(conversationId);
                if (timer) {
                    clearTimeout(timer);
                    this.debounceTimers.delete(conversationId);
                }
                // 清理 PlanData 缓存
                this.lastPlanDataCache.delete(conversationId);
                // 清理文件监听器
                const watcher = this.fileWatchers.get(conversationId);
                if (watcher) {
                    watcher.dispose();
                    this.fileWatchers.delete(conversationId);
                }
            }
        });
    }
    /**
     * 切换到 Markdown 源码编辑模式（双列）
     */
    async switchToMarkdownSource(document, webviewPanel) {
        try {
            // 先在旁边打开默认的文本编辑器（双列模式）
            await vscode.window.showTextDocument(document, {
                viewColumn: vscode.ViewColumn.Beside, // 在旁边打开
                preserveFocus: false,
                preview: false
            });
            // 然后关闭 Custom Editor
            // webviewPanel.dispose();
            logger_1.Logger.debug('Switched to Markdown source (side by side)');
        }
        catch (error) {
            logger_1.Logger.error('Failed to switch to Markdown source', error);
            vscode.window.showErrorMessage((0, i18n_1.t)('failedToSwitchToMarkdownSource'));
        }
    }
    /**
     * 切换回 Preview 模式（关闭源码编辑器，保留 Preview）
     */
    async switchToPreviewMode(document) {
        try {
            const targetUri = document.uri.toString();
            logger_1.Logger.debug('Switching to Preview mode', { uri: targetUri });
            // 使用 tabGroups API 查找并关闭源码编辑器的 tab
            const tabGroups = vscode.window.tabGroups.all;
            let closedCount = 0;
            for (const group of tabGroups) {
                for (const tab of group.tabs) {
                    // 检查是否是文本编辑器 tab（不是 Custom Editor）
                    if (tab.input instanceof vscode.TabInputText) {
                        const tabUri = tab.input.uri.toString();
                        // 匹配目标文档的源码编辑器
                        if (tabUri === targetUri) {
                            // 关闭这个 tab
                            await vscode.window.tabGroups.close(tab);
                            closedCount++;
                        }
                    }
                }
            }
            logger_1.Logger.debug('Preview mode switch completed', { closedTabs: closedCount });
        }
        catch (error) {
            logger_1.Logger.error('Failed to switch to Preview mode', error);
        }
    }
    /**
     * 处理 TODO 构建
     */
    async handleBuildPlan(id) {
        this.planActionReport('build', id);
        try {
            // 使用 CHAT_SEND_MESSAGE 命令发送消息到聊天区
            // 先忽略不同conversation的问题
            await vscode.commands.executeCommand('tencentcloud.codingcopilot.chat.sendMessage', {
                message: (0, i18n_1.t)('acceptPlanAndStart'),
                options: {
                    mode: 'plan'
                }
            });
        }
        catch (error) {
            logger_1.Logger.error('Failed to build plan', error);
        }
    }
    /**
     * 处理 Plan 内容统一更新
     * 包含 markdown 内容、设计配置和 TODO 列表的统一更新
     * 这个方法可能有隐患，当字符串内容很多时，可能会阻塞信道
     */
    async handlePlanContentUpdate(id, planResult) {
        try {
            if (!planResult) {
                planResult = await syncPlan_1.PlanDataLoader.loadPlan(this.context, id);
            }
            await vscode.commands.executeCommand('CodeBuddy.plan.markdownUpdate', planResult);
        }
        catch (error) {
            logger_1.Logger.error('Failed to update plan content', error);
        }
    }
    /**
     * 计算差异并通知
     * @param conversationId 对话ID
     * @returns diff 结果
     */
    async calculateAndNotifyDiff(conversationId) {
        try {
            const mainExtensionPath = this.getGenieExtensionPath();
            const diffResult = await (0, data_1.calculateCurrentDiff)(mainExtensionPath, conversationId);
            if (diffResult) {
                logger_1.Logger.debug('Calculated diff', { conversationId, stats: diffResult.stats });
            }
            return diffResult;
        }
        catch (error) {
            logger_1.Logger.error('Failed to calculate diff', error);
            return null;
        }
    }
    /**
     * 获取主扩展(tencent-cloud.coding-copilot)的globalStorage路径
     */
    getGenieExtensionPath() {
        const currentGlobalStoragePath = this.context.globalStorageUri.fsPath;
        const mainExtensionPath = currentGlobalStoragePath.replace(/vscode\.lexical-markdown-editor$/, 'tencent-cloud.coding-copilot');
        return mainExtensionPath;
    }
    /**
     * 设置文件监听器，监听plan.md的外部变化
     */
    async setupFileWatcher(conversationId, webviewPanel) {
        try {
            const mainExtensionPath = this.getGenieExtensionPath();
            // 构建 plan 目录的 URI（用于 RelativePattern）
            const planDirUri = vscode.Uri.file(`${mainExtensionPath}/plans/${conversationId}`);
            // 使用 RelativePattern 创建文件系统监听器
            // 监听文件变化和创建事件，忽略删除事件
            // 参数: (pattern, ignoreCreateEvents, ignoreChangeEvents, ignoreDeleteEvents)
            const watcher = vscode.workspace.createFileSystemWatcher(new vscode.RelativePattern(planDirUri, 'plan.md'), false, // ignoreCreateEvents - 监听创建事件
            false, // ignoreChangeEvents - 监听内容变化
            true // ignoreDeleteEvents - 忽略删除事件
            );
            // 创建 CompositeDisposable 来管理所有相关的 disposables
            const disposables = [];
            // 防抖处理的文件变化回调（只有内容真正变化时才更新）
            const handleFileChange = async () => {
                try {
                    // 读取 markdown 文件内容
                    const planResult = await (0, data_1.getLocalPlanResult)(mainExtensionPath, conversationId);
                    if (planResult.status === 'prepare') {
                        return;
                    }
                    // 比较 PlanData 是否真正变化
                    const cachedPlanData = this.lastPlanDataCache.get(conversationId);
                    const hasContentChanged = cachedPlanData?.content !== planResult.content;
                    const hasTodolistChanged = JSON.stringify(cachedPlanData?.todolist) !== JSON.stringify(planResult.todolist);
                    const hasDesignChanged = JSON.stringify(cachedPlanData?.design) !== JSON.stringify(planResult.design);
                    if (!hasContentChanged && !hasTodolistChanged && !hasDesignChanged) {
                        logger_1.Logger.debug('FileWatcher: no changes detected, skipping update (avoiding circular refresh)', { conversationId });
                        return;
                    }
                    logger_1.Logger.info('FileWatcher: changes detected from external source', {
                        conversationId,
                        hasContentChanged,
                        hasTodolistChanged,
                        hasDesignChanged
                    });
                    // 更新缓存
                    this.lastPlanDataCache.set(conversationId, planResult);
                    // 更新插件侧的planResult
                    await this.handlePlanContentUpdate(conversationId);
                    // 计算差异统计
                    const diffResult = await this.calculateAndNotifyDiff(conversationId);
                    await webviewPanel.webview.postMessage({
                        type: 'updateMarkdownContent',
                        content: planResult.content,
                        diffStats: diffResult?.stats,
                        diffChanges: diffResult?.changes
                    });
                }
                catch (error) {
                    logger_1.Logger.error('FileWatcher: failed to handle file change', error);
                }
            };
            // 监听文件内容变化（带防抖 + 内容比较）
            disposables.push(watcher.onDidChange(() => {
                // 清除之前的定时器
                const existingTimer = this.debounceTimers.get(conversationId);
                if (existingTimer) {
                    clearTimeout(existingTimer);
                }
                // 设置新的防抖定时器
                const timer = setTimeout(() => {
                    this.debounceTimers.delete(conversationId);
                    handleFileChange();
                }, LexicalMarkdownEditorProvider.DEBOUNCE_DELAY);
                this.debounceTimers.set(conversationId, timer);
            }));
            // 监听文件创建事件（文件创建时需要加载完整 plan 数据）
            disposables.push(watcher.onDidCreate(async () => {
                try {
                    // 文件创建时需要加载完整的 plan 数据
                    const planData = await syncPlan_1.PlanDataLoader.loadPlan(this.context, conversationId);
                    if (!planData || planData?.status === 'prepare') {
                        return;
                    }
                    // 更新 PlanData 缓存
                    this.lastPlanDataCache.set(conversationId, planData);
                    await webviewPanel.webview.postMessage({
                        type: 'reloadPlan',
                        planResult: planData
                    });
                }
                catch (error) {
                    logger_1.Logger.error('FileWatcher: failed to reload plan after creation', error);
                }
            }));
            // 添加 watcher 本身到 disposables
            disposables.push(watcher);
            // 保存组合的 disposable
            this.fileWatchers.set(conversationId, vscode.Disposable.from(...disposables));
            logger_1.Logger.info('File watcher set up', { conversationId, path: planDirUri.fsPath });
        }
        catch (error) {
            logger_1.Logger.error('Failed to setup file watcher', error);
        }
    }
    /**
     * 从文件路径提取 conversationId
     * 使用 path 模块解析路径，适配 Windows、Linux、macOS 等各种操作系统
     */
    extractConversationId(filePath) {
        // 路径格式: .../plans/{conversationId}/plan.md
        // 检查文件名是否为 plan.md
        if (path.basename(filePath) !== 'plan.md') {
            return null;
        }
        // 获取 plan.md 的父目录（即 conversationId 目录）
        const parentDir = path.dirname(filePath);
        const conversationId = path.basename(parentDir);
        // 验证父目录的父目录是否为 plans
        const grandParentDir = path.dirname(parentDir);
        if (path.basename(grandParentDir) !== 'plans') {
            return null;
        }
        // 验证 conversationId 格式（可选，增加安全性）
        if (!/^[a-zA-Z0-9_-]+$/.test(conversationId)) {
            return null;
        }
        return conversationId;
    }
    /**
     * 生成 WebView HTML
     */
    getHtmlForWebview(webview, document, planData, diffResult) {
        const scriptUri = webview.asWebviewUri(vscode.Uri.joinPath(this.context.extensionUri, 'webview', 'dist', 'editor.js'));
        // 使用 path.basename 获取文件名，适配各种操作系统
        const fileName = path.basename(document.fileName);
        // 获取并清除 scrollToPart 参数
        const scrollToPart = this.context.workspaceState.get('lexical.scrollToPart');
        if (scrollToPart) {
            this.context.workspaceState.update('lexical.scrollToPart', undefined);
        }
        return `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <meta http-equiv="Content-Security-Policy" content="default-src 'none'; style-src ${webview.cspSource} 'unsafe-inline'; script-src ${webview.cspSource} 'unsafe-inline';">
  <title>Lexical Markdown Editor</title>
  <style>
    body {
      margin: 0;
      padding: 0;
      overflow: hidden;
      background-color: var(--vscode-editor-background);
      color: var(--vscode-editor-foreground);
    }
    #root {
      width: 100vw;
      height: 100vh;
    }
  </style>
</head>
<body>
  <div id="root"></div>
  <script>
    // 初始化数据，供 React 应用使用
    window.__locale__ = ${JSON.stringify(vscode.env.language)};
    window.initialMarkdown = ${JSON.stringify(planData.content)};
    window.initialTodoList = ${JSON.stringify(planData.todolist)};
    window.initialDesignConfig = ${JSON.stringify(planData.design)};
    window.fileName = ${JSON.stringify(fileName)};
    window.planStatus = ${JSON.stringify(planData.status)};
    window.conversationId = ${JSON.stringify(planData.id)};
    window.planName = ${JSON.stringify(planData.name)};
    window.planHasConfirmed = ${JSON.stringify(planData.hasConfirmed)};
    window.scrollToPart = ${JSON.stringify(scrollToPart || null)};
    window.initialDiffStats = ${JSON.stringify(diffResult?.stats || null)};
    window.initialDiffChanges = ${JSON.stringify(diffResult?.changes || [])};
    window.planOverview = ${JSON.stringify(planData.overview)};
    window.parts = ${JSON.stringify(planData.parts)};
  </script>
  <script src="${scriptUri}"></script>
</body>
</html>`;
    }
    planActionReport(action, id) {
        const planResult = this.lastPlanDataCache.get(id);
        if (!planResult) {
            logger_1.Logger.warn('Plan action report: no cached plan data found', { id, action });
            return;
        }
        try {
            // 计算todo的数量情况，输出字符串为(进行中-取消-完成-未开始/总共)
            const statusOrder = ['in_progress', 'cancelled', 'completed', 'pending'];
            const counts = statusOrder.map(status => planResult.todolist.filter((todo) => todo.status === status).length);
            const todoStatus = `${counts.join('-')}/${planResult.todolist.length}`;
            vscode.commands.executeCommand('CodeBuddy.reportFromIDE', {
                key: 'ide_plan_action',
                content: {
                    action,
                    conversationId: planResult.id,
                    name: planResult.name,
                    status: planResult.status,
                    parts: planResult.parts.join(","),
                    todoStatus
                }
            });
        }
        catch (error) {
            logger_1.Logger.error('Plan action report failed', error);
        }
    }
}
// 存储 conversationId 到 webviewPanel 的映射（改为 public 以便外部访问）
LexicalMarkdownEditorProvider.webviewPanels = new Map();
// 防抖延迟时间（毫秒）
LexicalMarkdownEditorProvider.DEBOUNCE_DELAY = 300;
//# sourceMappingURL=extension.js.map